package com.trafficparrot.examples.http;

import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;
import org.json.JSONArray;
import org.json.JSONObject;

import java.io.IOException;

import static org.hamcrest.CoreMatchers.is;
import static org.assertj.core.api.Assertions.assertThat;

public class HttpClient {

    private final String virtualService;

    public HttpClient(String virtualService) {
        this.virtualService = virtualService;
    }

    public String post(String path, String requestBody) throws IOException {
        try (CloseableHttpClient httpclient = HttpClients.createDefault()) {
            HttpPost request = new HttpPost(virtualService + path);
            request.setEntity(new StringEntity(requestBody));
            try (CloseableHttpResponse response = httpclient.execute(request)) {
                int statusCode = response.getStatusLine().getStatusCode();
                String responseBody = EntityUtils.toString(response.getEntity());
                assertThat(statusCode).describedAs(responseBody).isEqualTo(200);
                return responseBody;
            }
        }
    }

    public String get(String path) throws IOException {
        return get(path, 200);
    }

    public String get(String path, int expectedStatus) throws IOException {
        try (CloseableHttpClient httpclient = HttpClients.createDefault()) {
            HttpGet request = new HttpGet(virtualService + path);
            try (CloseableHttpResponse response = httpclient.execute(request)) {
                int statusCode = response.getStatusLine().getStatusCode();
                String responseBody = EntityUtils.toString(response.getEntity());
                assertThat(statusCode).describedAs(responseBody).isEqualTo(expectedStatus);
                return responseBody;
            }
        }
    }

    /**
     * For example purposes, print out the mapping content as understood by Traffic Parrot
     *
     * @param content Filter the mappings by the ones that contain this content
     */
    public void displayHttpMappingsContaining(String content) throws IOException {
        System.out.println("Mappings containing '" + content + "':");
        String body = get("/__admin");

        JSONObject json = new JSONObject(body);
        JSONArray mappings = json.getJSONArray("mappings");
        for (Object mapping : mappings) {
            JSONObject jsonObject = (JSONObject) mapping;
            String mappingString = jsonObject.toString(4);
            if (mappingString.contains(content) && !mappingString.contains("GRPC")) {
                System.out.println(mappingString);
            }
        }
    }
}
