package com.trafficparrot.examples.http;

import com.trafficparrot.examples.tools.http.HttpClient;
import org.apache.http.MalformedChunkCodingException;
import org.apache.http.NoHttpResponseException;
import org.apache.http.ProtocolException;
import org.apache.http.client.ClientProtocolException;
import org.junit.Ignore;
import org.junit.Test;

import java.io.IOException;
import java.time.Duration;

import static org.assertj.core.api.Assertions.assertThat;
import static org.assertj.core.api.Assertions.fail;

public class SimulateHttpFaultsIntegrationTest {
    private final HttpClient httpClient = new HttpClient("http://localhost:18081");

    @Test(timeout = 30_000)
    public void dripBytesEvery1s() throws IOException {
        httpClient.displayHttpMappingsContaining("dripBytesEvery1s");

        long start = System.nanoTime();

        String response = httpClient.get("/dripBytesEvery1s");

        long end = System.nanoTime();

        assertThat(response).isEqualTo("12");
        assertThat(Duration.ofNanos(end - start)).isGreaterThan(Duration.ofSeconds(2));
    }

    @Test(timeout = 30_000)
    public void dripBytesEvery100ms() throws IOException {
        httpClient.displayHttpMappingsContaining("dripBytesEvery100ms");

        long start = System.nanoTime();

        String response = httpClient.get("/dripBytesEvery100ms");

        long end = System.nanoTime();

        assertThat(response).isEqualTo("1234");
        assertThat(Duration.ofNanos(end - start)).isGreaterThan(Duration.ofMillis(400));
    }

    @Test(timeout = 30_000)
    public void closeConnectionAfterHalfResponse() throws IOException {
        httpClient.displayHttpMappingsContaining("closeConnectionAfterHalfResponse");

        String response = httpClient.get("/closeConnectionAfterHalfResponse");

        assertThat(response).isEqualTo("First half th");
    }

    @Test(timeout = 30_000)
    public void emptyResponse() throws IOException {
        httpClient.displayHttpMappingsContaining("emptyResponse");

        try {
            httpClient.get("/emptyResponse");
            fail("Expected NoHttpResponseException");
        } catch (NoHttpResponseException e) {
            assertThat(e).hasMessageContaining("failed to respond");
        }
    }

    @Test(timeout = 30_000)
    public void malformedResponseChunk() throws IOException {
        httpClient.displayHttpMappingsContaining("malformedResponseChunk");

        try {
            httpClient.get("/malformedResponseChunk");
            fail("Expected MalformedChunkCodingException");
        } catch (MalformedChunkCodingException e) {
            assertThat(e).hasMessageContaining("Bad chunk header");
        }
    }

    @Test(timeout = 30_000)
    public void incorrectSmallContentLength() throws IOException {
        httpClient.displayHttpMappingsContaining("incorrectSmallContentLength");

        String response = httpClient.get("/incorrectSmallContentLength", 500);

        assertThat(response).contains("java.io.IOException: written 17 &gt; 1 content-length");
    }

    @Test(timeout = 30_000)
    public void randomDataThenClose() throws IOException {
        httpClient.displayHttpMappingsContaining("randomDataThenClose");

        try {
            httpClient.get("/randomDataThenClose");
            fail("Expected ClientProtocolException");
        } catch (ClientProtocolException e) {
            assertThat(e).hasCauseInstanceOf(ProtocolException.class);
        }
    }
}
