package com.trafficparrot.examples.tools.grpc;

import com.google.common.net.HostAndPort;
import com.trafficparrot.examples.grpc.Item;
import com.trafficparrot.examples.grpc.OrderGrpc;
import com.trafficparrot.examples.grpc.OrderStatus;
import io.grpc.ManagedChannel;
import io.grpc.netty.NettyChannelBuilder;
import io.netty.handler.ssl.SslContext;

import java.util.Optional;
import java.util.concurrent.TimeUnit;

import static io.grpc.netty.NegotiationType.PLAINTEXT;
import static io.grpc.netty.NegotiationType.TLS;

public class OrderClient {

    private final ManagedChannel channel;
    private final OrderGrpc.OrderBlockingStub orderService;

    private OrderClient(ManagedChannel channel, OrderGrpc.OrderBlockingStub orderService) {
        this.channel = channel;
        this.orderService = orderService;
    }

    public static OrderClient createTlsOrderClient(HostAndPort hostAndPort, SslContext sslContext, String overrideAuthority) {
        return createOrderClient(hostAndPort, Optional.of(sslContext), overrideAuthority);
    }

    public static OrderClient createNonTlsOrderClient(HostAndPort hostAndPort) {
        return createOrderClient(hostAndPort, Optional.empty(), "localhost");
    }

    private static OrderClient createOrderClient(HostAndPort hostAndPort, Optional<SslContext> sslContext, String overrideAuthority) {
        NettyChannelBuilder builder = NettyChannelBuilder.forAddress(hostAndPort.getHost(), hostAndPort.getPort())
                .overrideAuthority(overrideAuthority)
                .negotiationType(sslContext.map(x -> TLS).orElse(PLAINTEXT));
        sslContext.ifPresent(builder::sslContext);
        ManagedChannel channel = builder.build();

        OrderGrpc.OrderBlockingStub orderService = OrderGrpc.newBlockingStub(channel);
        return new OrderClient(channel, orderService);
    }

    public void shutdown() throws InterruptedException {
        channel.shutdown().awaitTermination(5, TimeUnit.SECONDS);
    }

    public OrderStatus order(int quantity, int sku) {
        return orderService.purchase(Item.newBuilder().setQuantity(quantity).setSku(sku).build());
    }
}

