package com.trafficparrot.sdk.example.jms;

import com.github.tomakehurst.wiremock.extension.Parameters;
import com.trafficparrot.sdk.jms.Destination;
import com.trafficparrot.sdk.jms.JmsResponse;
import org.junit.Test;

import javax.jms.BytesMessage;
import javax.jms.TextMessage;

import static com.trafficparrot.sdk.jms.DestinationType.QUEUE;
import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

public class ChooseResponseQueueBasedOnRequestMessageBodyTest {
    private static final Destination REQUEST_DESTINATION = new Destination("request_queue", QUEUE);
    private static final Destination RESPONSE_DESTINATION = new Destination("response_queue", QUEUE);
    private static final String RESPONSE_TEXT = "response_text";
    private static final JmsResponse RESPONSE = new JmsResponse(RESPONSE_DESTINATION, RESPONSE_TEXT);
    private final TextMessage requestMessage = mock(TextMessage.class);
    private final ChooseResponseQueueBasedOnRequestMessageBody chooseResponseQueue = new ChooseResponseQueueBasedOnRequestMessageBody();

    @Test
    public void redirectsToResponseQueue1WhenRequestContainsFoo() throws Exception {
        when(requestMessage.getText()).thenReturn("{\"requestField\": \"foo\"}");

        JmsResponse response = chooseResponseQueue.transform(REQUEST_DESTINATION, requestMessage, RESPONSE, new Parameters());

        assertThat(response.destination.name).isEqualTo("response_queue_1");
    }

    @Test
    public void redirectsToResponseQueue2WhenRequestContainsBar() throws Exception {
        when(requestMessage.getText()).thenReturn("{\"requestField\": \"bar\"}");

        JmsResponse response = chooseResponseQueue.transform(REQUEST_DESTINATION, requestMessage, RESPONSE, new Parameters());

        assertThat(response.destination.name).isEqualTo("response_queue_2");
    }

    @Test
    public void redirectsToResponseOriginalQueueWhenRequestFiledValueIsNotMapped() throws Exception {
        when(requestMessage.getText()).thenReturn("{\"requestField\": \"unknown\"}");

        JmsResponse response = chooseResponseQueue.transform(REQUEST_DESTINATION, requestMessage, RESPONSE, new Parameters());

        assertThat(response.destination).isSameAs(RESPONSE_DESTINATION);
    }

    @Test
    public void redirectsToResponseOriginalQueueWhenRequestIsNotJson() throws Exception {
        when(requestMessage.getText()).thenReturn("this is not json");

        JmsResponse response = chooseResponseQueue.transform(REQUEST_DESTINATION, requestMessage, RESPONSE, new Parameters());

        assertThat(response.destination).isSameAs(RESPONSE_DESTINATION);
    }

    @Test
    public void skipsNonTestMessages() throws Exception {
        JmsResponse response = chooseResponseQueue.transform(new Destination("another_queue", QUEUE), mock(BytesMessage.class), RESPONSE, new Parameters());

        assertThat(response).isSameAs(RESPONSE);
    }
}