package com.trafficparrot.examples.tools.jms;

import org.apache.activemq.ActiveMQConnectionFactory;

import javax.jms.*;
import java.io.*;
import java.lang.IllegalStateException;
import java.util.concurrent.CountDownLatch;

public class JmsClient {
    private final String connection;

    public JmsClient(String connection) {
        this.connection = connection;
    }

    public void sendObjectMessageToQueue(String queueName, Object object) throws JMSException {
        sendMessageTo(true, queueName, object);
    }

    public void sendTextMessageToQueue(String queueName, String text) throws JMSException {
        sendMessageTo(true, queueName, text);
    }

    public String consumeTextMessageFromQueue(String queueName) throws JMSException {
        return consumeTextMessage(queueName);
    }

    public void sendTextMessageToTopic(String topicName, String text) throws JMSException {
        sendMessageTo(false, topicName, text);
    }

    public WaitForOneJmsMessage listenOnTopic(String topic) throws JMSException {
        ActiveMQConnectionFactory connectionFactory = new ActiveMQConnectionFactory(connection);
        Connection connection = connectionFactory.createConnection();
        return listenOnTopic(topic, connection);
    }

    public WaitForOneJmsMessage listenOnTopic(String topic, Connection connection) throws JMSException {
        connection.setExceptionListener(new PrintStackTraceExceptionListener());
        connection.start();
        Session session = connection.createSession(false, Session.AUTO_ACKNOWLEDGE);
        MessageConsumer consumer = session.createConsumer(session.createTopic(topic));
        WaitForOneJmsMessage listener = new WaitForOneJmsMessage(new CountDownLatch(1), connection, session, consumer);
        consumer.setMessageListener(listener);
        return listener;
    }

    private void sendMessageTo(boolean queue, String name, Object object) throws JMSException {
        ActiveMQConnectionFactory connectionFactory = new ActiveMQConnectionFactory(connection);
        try (Connection connection = connectionFactory.createConnection()) {
            connection.setExceptionListener(new PrintStackTraceExceptionListener());
            connection.start();
            try (Session session = connection.createSession(false, Session.AUTO_ACKNOWLEDGE)) {
                BytesMessage message = session.createBytesMessage();
                message.writeBytes(javaObjectBytes(object));
                Destination destination = destination(queue, name, session);
                try (MessageProducer producer = session.createProducer(destination)) {
                    producer.setDeliveryMode(DeliveryMode.PERSISTENT);
                    producer.send(message);
                }
            }
        }
    }

    private void sendMessageTo(boolean queue, String name, String text) throws JMSException {
        ActiveMQConnectionFactory connectionFactory = new ActiveMQConnectionFactory(connection);
        try (Connection connection = connectionFactory.createConnection()) {
            connection.setExceptionListener(new PrintStackTraceExceptionListener());
            connection.start();
            try (Session session = connection.createSession(false, Session.AUTO_ACKNOWLEDGE)) {
                Message message = session.createTextMessage(text);
                Destination destination = destination(queue, name, session);
                try (MessageProducer producer = session.createProducer(destination)) {
                    producer.setDeliveryMode(DeliveryMode.PERSISTENT);
                    producer.send(message);
                }
            }
        }
    }

    private static Destination destination(boolean queue, String name, Session session) throws JMSException {
        if (queue) {
            return session.createQueue(name);
        } else {
            return session.createTopic(name);
        }
    }

    private String consumeTextMessage(String queueName) throws JMSException {
        Message message = consumeMessage(queueName);
        TextMessage textMessage = (TextMessage) message;
        return textMessage.getText();
    }

    public Object consumeObjectMessage(String queueName) throws JMSException, IOException, ClassNotFoundException {
        Message message = consumeMessage(queueName);
        BytesMessage bytesMessage = (BytesMessage) message;
        byte[] data = new byte[(int) bytesMessage.getBodyLength()];
        bytesMessage.readBytes(data);
        try (ObjectInputStream objectInputStream = new ObjectInputStream(new ByteArrayInputStream(data))) {
            return objectInputStream.readObject();
        }
    }

    private Message consumeMessage(String queueName) throws JMSException {
        ActiveMQConnectionFactory connectionFactory = new ActiveMQConnectionFactory(connection);
        try (Connection connection = connectionFactory.createConnection()) {
            connection.setExceptionListener(new PrintStackTraceExceptionListener());
            connection.start();
            try (Session session = connection.createSession(false, Session.AUTO_ACKNOWLEDGE);
                MessageConsumer consumer = session.createConsumer(session.createQueue(queueName))) {
                return consumer.receive();
            }
        }
    }

    private byte[] javaObjectBytes(Object object) {
        try (ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
             ObjectOutputStream objectOutputStream = new ObjectOutputStream(byteArrayOutputStream)) {
            objectOutputStream.writeObject(object);
            return byteArrayOutputStream.toByteArray();
        } catch (IOException e) {
            throw new IllegalStateException(e);
        }
    }

    private static class PrintStackTraceExceptionListener implements ExceptionListener {

        @Override
        public void onException(JMSException exception) {
            exception.printStackTrace();
        }
    }
}
