package com.trafficparrot.sdk.example.http;

import com.github.tomakehurst.wiremock.common.FileSource;
import com.github.tomakehurst.wiremock.extension.Parameters;
import com.github.tomakehurst.wiremock.http.HttpHeader;
import com.github.tomakehurst.wiremock.http.Request;
import com.github.tomakehurst.wiremock.http.ResponseDefinition;
import com.trafficparrot.sdk.http.HttpResponseDefinitionTransformer;
import com.trafficparrot.sdk.http.TransformedRequest;
import com.trafficparrot.sdk.http.TransformedRequestDefinition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.xml.sax.SAXException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

import static com.github.tomakehurst.wiremock.http.HttpHeader.httpHeader;
import static com.trafficparrot.sdk.http.TransformedRequest.like;

/**
 * To use it create mapping move-dates-in-xml-body-a5b59b84-d8ae-4050-ae15-e9635d1f0d2f.json:
 *
 * {
 *   "id" : "a5b59b84-d8ae-4050-ae15-e9635d1f0d2f",
 *   "name" : "move-dates-in-xml-body-a5b59b84-d8ae-4050-ae15-e9635d1f0d2f.json",
 *   "request" : {
 *     "urlPattern" : ".*",
 *     "method" : "ANY"
 *   },
 *   "response" : {
 *     "status" : 200,
 *     "proxyBaseUrl" : "http://localhost:9999/",
 *     "transformers" : [ "MoveDatesInXmlRequestTransformer", "MoveDatesInXmlResponseTransformer" ]
 *   },
 *   "uuid" : "a5b59b84-d8ae-4050-ae15-e9635d1f0d2f"
 * }
 *
 * To configure it, set in trafficparrot.properties:
 *
 * trafficparrot.http.responsetransformers=com.trafficparrot.sdk.example.http.MoveDatesInXmlRequestTransformer,com.trafficparrot.sdk.example.http.MoveDatesInXmlResponseTransformer
 * trafficparrot.http.soap.message.signer.enabled=false
 * trafficparrot.http.handlebars.enabled=false
 * trafficparrot.http.handlebars.helpers=
 *
 * It accepts request body:
 *
 * <ns:transactions xmlns:ns="http://example.trafficparrot.com"><get><startDate>2020-01-03</startDate></get></ns:transactions>
 */
public class MoveDatesInXmlRequestTransformer extends HttpResponseDefinitionTransformer {
    private static final Logger LOGGER = LoggerFactory.getLogger(MoveDatesInXmlRequestTransformer.class);

    public static final long FOUR_YEARS_IN_MILLIS = 4L * 365 * 24 * 60 * 60 * 1000;
    public static final HttpHeader TRANSFORMER_ADDED_REQUEST_HEADER = httpHeader("transformed-and-proxied-by", MoveDatesInXmlRequestTransformer.class.getName());

    @Override
    public ResponseDefinition doTransform(Request originalRequest, ResponseDefinition originalResponseDefinition, FileSource fileSource, Parameters parameters) {
        LOGGER.info("Transformer received request " + originalRequest);
        LOGGER.info("Original request body " + originalRequest.getBodyAsString());
        TransformedRequest transformedRequestToProxy = like(originalRequest)
                .withTransformBody(this::moveDatesInRequestBody)
                .withHeader(TRANSFORMER_ADDED_REQUEST_HEADER);
        LOGGER.info("Transformed request to proxy body " + transformedRequestToProxy.getBodyAsString());
        TransformedRequestDefinition transformedRequestDefinition = new TransformedRequestDefinition(originalResponseDefinition, transformedRequestToProxy);
        LOGGER.info("Transformer returning definition " + transformedRequestDefinition);
        return transformedRequestDefinition;
    }

    private String moveDatesInRequestBody(String requestMessageBody) {
        try {
            // parse the XML request body
            DocumentBuilderFactory abstractFactory = DocumentBuilderFactory.newInstance();
            DocumentBuilder factory = abstractFactory.newDocumentBuilder();
            Document doc = factory.parse(new ByteArrayInputStream(requestMessageBody.getBytes()));
            String requestDateString = XPathFactory.newInstance().newXPath().compile("/transactions/get/startDate/text()").evaluate(doc);
            SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
            Date requestDate = format.parse(requestDateString);

            // calculate the new date (in the past)
            Date movedDate = new Date(requestDate.getTime() - FOUR_YEARS_IN_MILLIS);
            String movedDateString = format.format(movedDate);

            // set new date
            return requestMessageBody.replace(requestDateString, movedDateString);
        } catch (ParserConfigurationException | SAXException | IOException | XPathExpressionException | ParseException e) {
            throw new IllegalArgumentException(e);
        }
    }

    @Override
    public String getName() {
        return getClass().getSimpleName();
    }
}
